using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using DataContracts;
using System.Runtime.CompilerServices;

namespace Ato.EN.IntegrationServices.CodeGenerationCHTWTHHLDBEN
{

    public partial class CHTWTHHLDBEN2018Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="CHTWTHHLDBEN2018Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public CHTWTHHLDBEN2018Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return ((from x in nodes select x).Distinct().Count() == nodes.Count());
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
           if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }
				//The logic in After function expects "---" for day and "--" for month. 
				//Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
					return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length < 13 || abn.Length < 11)
                {
                    response = false;
                }
                else
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 2), out numeric))
                        response = false;
                    else if (Regex.IsMatch(usi, @"^[a-zA-Z]{3}\d{4}[a-zA-Z]{2}"))
                        response = false;
                    else
                        response = true;
                }
            }
            return response;
        }


        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public CHTWTHHLDBEN2018 ConsumedReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(CHTWTHHLDBEN2018 report)
        {

            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = report;

            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------

    
            #region VR.ATO.CHTWTHHLDBEN.000019
    
            /*  VR.ATO.CHTWTHHLDBEN.000019
            Beneficiary First name has a maximum field length 15
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonNameDetails.GivenName.Text) > 15
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN24) > 15
    
            Data Elements:
    
            ^CHTWTHHLDBEN24 = Beneficiary:PersonNameDetails.GivenName.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN24) > 15);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000019", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary first name must not exceed 15 characters",
                    LongDescription = @"For the Beneficiary first name, the maximum allowable characters for this form is 15",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000019"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN24", Value = report.CHTWTHHLDBEN24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000019
    
            #region VR.ATO.CHTWTHHLDBEN.000022
    
            /*  VR.ATO.CHTWTHHLDBEN.000022
            Beneficiary Postal Suburb/Town is mandatory
    
            Legacy Rule Format:
            IF [chtwthhldben64] = NULLORBLANK
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben64] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.LocalityName.Text

            Technical Business Rule Format:
            ^CHTWTHHLDBEN64 = BLANK
    
            Data Elements:
    
            ^CHTWTHHLDBEN64 = Beneficiary.Postal:AddressDetails.LocalityName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN64) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430245", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City must be present in an address",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000022"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN64", Value = report.CHTWTHHLDBEN64 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000022
    
            #region VR.ATO.CHTWTHHLDBEN.000023
    
            /*  VR.ATO.CHTWTHHLDBEN.000023
            Beneficiary Entity type is mandatory
    
            Legacy Rule Format:
            IF [chtwthhldben21] = NULL
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben21] = CHTWTHHLDBEN:Beneficiary:rvctc1.02.04:IncomeTax.BeneficiaryEntityType.Code

            Technical Business Rule Format:
            ^CHTWTHHLDBEN21 = NULL
    
            Data Elements:
    
            ^CHTWTHHLDBEN21 = Beneficiary:IncomeTax.BeneficiaryEntityType.Code
            */
            assertion = (report.CHTWTHHLDBEN21 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary entity type is required",
                    LongDescription = @"The valid values for the Beneficiary entity type are:  'I' - Individual, 'T' - Trust, 'C' - Company, 'P' - Partnership, 'F' - Fund, 'S' - SMSF",
                    Location = "/xbrli:xbrl/tns:IncomeTax.BeneficiaryEntityType.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000023"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN21", Value = report.CHTWTHHLDBEN21 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000023
    
            #region VR.ATO.CHTWTHHLDBEN.000024
    
            /*  VR.ATO.CHTWTHHLDBEN.000024
            Beneficiary Postal address line 1 field is mandatory
    
            Legacy Rule Format:
            IF [chtwthhldben62] = NULLORBLANK
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben62] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.Line1.Text

            Technical Business Rule Format:
            ^CHTWTHHLDBEN62 = BLANK
    
            Data Elements:
    
            ^CHTWTHHLDBEN62 = Beneficiary.Postal:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN62) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410214", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000024"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN62", Value = report.CHTWTHHLDBEN62 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000024
    
            #region VR.ATO.CHTWTHHLDBEN.000025
    
            /*  VR.ATO.CHTWTHHLDBEN.000025
            Field contains all numeric characters with any combination of spaces; at least one alpha must be present.
    
            Legacy Rule Format:
            IF ([chtwthhldben32] <> NULLORBLANK) AND ([chtwthhldben32] DOES NOT CONTAIN SET("a-z","A-Z"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben32] = CHTWTHHLDBEN:Beneficiary:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            (^CHTWTHHLDBEN32 <> BLANK) AND (NotContainsSet(^CHTWTHHLDBEN32, '"a-z","A-Z"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN32 = Beneficiary:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN32) != true && !(IsMatch(report.CHTWTHHLDBEN32, @"[a-z]|[A-Z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000025", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary non-individual name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000025"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN32", Value = report.CHTWTHHLDBEN32 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000025
    
            #region VR.ATO.CHTWTHHLDBEN.000029
    
            /*  VR.ATO.CHTWTHHLDBEN.000029
            Beneficiary Non-Individual Name has a maximum field length 76
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text) > 76
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN32) > 76
    
            Data Elements:
    
            ^CHTWTHHLDBEN32 = Beneficiary:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN32) > 76);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary non-individual name must not exceed 76 characters",
                    LongDescription = @"For the Beneficiary non-individual name, the maximum allowable characters for this form is 76",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000029"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN32", Value = report.CHTWTHHLDBEN32 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000029
    
            #region VR.ATO.CHTWTHHLDBEN.000030
    
            /*  VR.ATO.CHTWTHHLDBEN.000030
            Beneficiary Postal Suburb/Town has a maximum field length 27
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.LocalityName.Text) > 27
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN69) > 27
    
            Data Elements:
    
            ^CHTWTHHLDBEN69 = Beneficiary.Business:AddressDetails.LocalityName.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN69) > 27);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary suburb/town must not exceed 27 characters",
                    LongDescription = @"For each Beneficiary suburb/town field, the maximum allowable characters for this form is 27",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000030"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN69", Value = report.CHTWTHHLDBEN69 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000030
    
            #region VR.ATO.CHTWTHHLDBEN.000030
    
            /*  VR.ATO.CHTWTHHLDBEN.000030
            Beneficiary Postal Suburb/Town has a maximum field length 27
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.LocalityName.Text) > 27
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN64) > 27
    
            Data Elements:
    
            ^CHTWTHHLDBEN64 = Beneficiary.Postal:AddressDetails.LocalityName.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN64) > 27);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary suburb/town must not exceed 27 characters",
                    LongDescription = @"For each Beneficiary suburb/town field, the maximum allowable characters for this form is 27",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000030"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN64", Value = report.CHTWTHHLDBEN64 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000030
    
            #region VR.ATO.CHTWTHHLDBEN.000030
    
            /*  VR.ATO.CHTWTHHLDBEN.000030
            Beneficiary Postal Suburb/Town has a maximum field length 27
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.LocalityName.Text) > 27
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN41) > 27
    
            Data Elements:
    
            ^CHTWTHHLDBEN41 = Beneficiary.Residential:AddressDetails.LocalityName.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN41) > 27);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary suburb/town must not exceed 27 characters",
                    LongDescription = @"For each Beneficiary suburb/town field, the maximum allowable characters for this form is 27",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000030"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN41", Value = report.CHTWTHHLDBEN41 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000030
    
            #region VR.ATO.CHTWTHHLDBEN.000032
    
            /*  VR.ATO.CHTWTHHLDBEN.000032
            Beneficiary Family name has a maximum field length 30
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonNameDetails.FamilyName.Text) > 30
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN23) > 30
    
            Data Elements:
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN23) > 30);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000032", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary family name must not exceed 30 characters",
                    LongDescription = @"For the Beneficiary family name, the maximum allowable characters for this form is 30",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000032"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000032
    
            #region VR.ATO.CHTWTHHLDBEN.000034
    
            /*  VR.ATO.CHTWTHHLDBEN.000034
            Family name or non-individual name must be present.
    
            Legacy Rule Format:
            IF ([chtwthhldben23] = NULLORBLANK) AND ([chtwthhldben32] = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben23] = CHTWTHHLDBEN:Beneficiary:pyde.02.00:PersonNameDetails.FamilyName.Text
            [chtwthhldben32] = CHTWTHHLDBEN:Beneficiary:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            (^CHTWTHHLDBEN23 = BLANK) AND (^CHTWTHHLDBEN32 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
    
            ^CHTWTHHLDBEN32 = Beneficiary:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN23) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN32) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000034", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary family name or Beneficiary non-individual name must be present",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000034"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN32", Value = report.CHTWTHHLDBEN32 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000034
    
            #region VR.ATO.CHTWTHHLDBEN.000035
    
            /*  VR.ATO.CHTWTHHLDBEN.000035
            Family name and non-individual name are both present. Only individual or non-individual name may be present.
    
            Legacy Rule Format:
            IF ([chtwthhldben23] <> NULLORBLANK) AND ([chtwthhldben32] <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben23] = CHTWTHHLDBEN:Beneficiary:pyde.02.00:PersonNameDetails.FamilyName.Text
            [chtwthhldben32] = CHTWTHHLDBEN:Beneficiary:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            (^CHTWTHHLDBEN23 <> BLANK) AND (^CHTWTHHLDBEN32 <> BLANK)
    
            Data Elements:
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
    
            ^CHTWTHHLDBEN32 = Beneficiary:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN23) != true && string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN32) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Both Beneficiary family name and Benficiary non-individual name present",
                    LongDescription = @"Beneficiary family name and Beneficiary non-individual name are both present. Only Beneficiary family name or Beneficiary non-individual name may be present.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000035"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN32", Value = report.CHTWTHHLDBEN32 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000035
    
            #region VR.ATO.CHTWTHHLDBEN.000041
    
            /*  VR.ATO.CHTWTHHLDBEN.000041
            Individual entities must complete family name/surname field.
    
            Legacy Rule Format:
            IF ([chtwthhldben21] = "I") AND ([chtwthhldben23] = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben21] = CHTWTHHLDBEN:Beneficiary:rvctc1.02.04:IncomeTax.BeneficiaryEntityType.Code
            [chtwthhldben23] = CHTWTHHLDBEN:Beneficiary:pyde.02.00:PersonNameDetails.FamilyName.Text

            Technical Business Rule Format:
            (^CHTWTHHLDBEN21 = 'I') AND (^CHTWTHHLDBEN23 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLDBEN21 = Beneficiary:IncomeTax.BeneficiaryEntityType.Code
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.CHTWTHHLDBEN21 == @"I" && string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN23) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000041", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary family name is required",
                    LongDescription = @"Individual entities must complete Beneficiary family name field.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.BeneficiaryEntityType.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000041"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN21", Value = report.CHTWTHHLDBEN21 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000041
    
            #region VR.ATO.CHTWTHHLDBEN.000042
    
            /*  VR.ATO.CHTWTHHLDBEN.000042
            Non-individual entities must complete non-individual name field.
    
            Legacy Rule Format:
            IF ([chtwthhldben21] <> "I") AND ([chtwthhldben32] = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben21] = CHTWTHHLDBEN:Beneficiary:rvctc1.02.04:IncomeTax.BeneficiaryEntityType.Code
            [chtwthhldben32] = CHTWTHHLDBEN:Beneficiary:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            (^CHTWTHHLDBEN21 <> 'I') AND (^CHTWTHHLDBEN32 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLDBEN21 = Beneficiary:IncomeTax.BeneficiaryEntityType.Code
    
            ^CHTWTHHLDBEN32 = Beneficiary:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.CHTWTHHLDBEN21 != @"I" && string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN32) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000042", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary non-individual name is required",
                    LongDescription = @"Non-individual entities must complete Beneficiary non-individual name field.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.BeneficiaryEntityType.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000042"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN21", Value = report.CHTWTHHLDBEN21 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN32", Value = report.CHTWTHHLDBEN32 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000042
    
            #region VR.ATO.CHTWTHHLDBEN.000046
    
            /*  VR.ATO.CHTWTHHLDBEN.000046
            The date entered is a future date. Date of birth cannot be any day in the future.
    
            Legacy Rule Format:
            IF ([chtwthhldben26] <> NULL) AND ([chtwthhldben26] > DATE(TODAY))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben26] = CHTWTHHLDBEN:Beneficiary:pyde.02.00:PersonDemographicDetails.Birth.Date

            Technical Business Rule Format:
            (^CHTWTHHLDBEN26 <> NULL) AND (^CHTWTHHLDBEN26 > Today())
    
            Data Elements:
    
            ^CHTWTHHLDBEN26 = Beneficiary:PersonDemographicDetails.Birth.Date
            */
            assertion = (report.CHTWTHHLDBEN26 != null && report.CHTWTHHLDBEN26.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000046", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary date of birth must be less than or equal to today's date",
                    LongDescription = @"The date entered is a future date. Beneficiary date of birth cannot be any day in the future.",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Birth.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000046"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN26", Value = GetValueOrEmpty(report.CHTWTHHLDBEN26) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000046
    
            #region VR.ATO.CHTWTHHLDBEN.000047
    
            /*  VR.ATO.CHTWTHHLDBEN.000047
            Amount withheld and amount of payments for the beneficiary must be present for the annual report.
    
            Legacy Rule Format:
            IF ([chtwthhldben45] = NULL) OR ([chtwthhldben46] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben45] = CHTWTHHLDBEN:Beneficiary:rvctc2.02.12:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount
            [chtwthhldben46] = CHTWTHHLDBEN:Beneficiary:rvctc2.02.12:IncomeTax.PayAsYouGoWithholding.PaymentGross.Amount

            Technical Business Rule Format:
            (^CHTWTHHLDBEN45 = NULL) OR (^CHTWTHHLDBEN46 = NULL)
    
            Data Elements:
    
            ^CHTWTHHLDBEN45 = Beneficiary:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount
    
            ^CHTWTHHLDBEN46 = Beneficiary:IncomeTax.PayAsYouGoWithholding.PaymentGross.Amount
            */
            assertion = (report.CHTWTHHLDBEN45 == null || report.CHTWTHHLDBEN46 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000047", Severity = ProcessMessageSeverity.Error,
                    Description = @"Amount withheld and Amount of payments for the beneficiary must be present",
                    LongDescription = @"Amount withheld and Amount of payments for the beneficiary must be present for the Annual TFN withholding report.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000047"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN45", Value = report.CHTWTHHLDBEN45.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN46", Value = report.CHTWTHHLDBEN46.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000047
    
            #region VR.ATO.CHTWTHHLDBEN.000049
    
            /*  VR.ATO.CHTWTHHLDBEN.000049
            Address details must be complete with street name and number, suburb/town, State, and postcode.
    
            Legacy Rule Format:
            IF (([chtwthhldben39] <> NULLORBLANK) OR ([chtwthhldben40] <> NULLORBLANK) OR ([chtwthhldben41] <> NULLORBLANK) OR ([chtwthhldben42] <> NULL) OR ([chtwthhldben43] <> NULL))
            AND ((([chtwthhldben39] = NULLORBLANK) AND ([chtwthhldben40] = NULLORBLANK)) OR ([chtwthhldben41] = NULLORBLANK) OR ([chtwthhldben42] = NULL) OR ([chtwthhldben43] = NULL))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben39] = CHTWTHHLDBEN:Beneficiary.Residential:pyde.02.00:AddressDetails.Line1.Text
            [chtwthhldben40] = CHTWTHHLDBEN:Beneficiary.Residential:pyde.02.00:AddressDetails.Line2.Text
            [chtwthhldben41] = CHTWTHHLDBEN:Beneficiary.Residential:pyde.02.00:AddressDetails.LocalityName.Text
            [chtwthhldben42] = CHTWTHHLDBEN:Beneficiary.Residential:pyde.02.00:AddressDetails.StateOrTerritory.Code
            [chtwthhldben43] = CHTWTHHLDBEN:Beneficiary.Residential:pyde.02.00:AddressDetails.Postcode.Text

            Technical Business Rule Format:
            ((^CHTWTHHLDBEN39 <> BLANK) OR (^CHTWTHHLDBEN40 <> BLANK) OR (^CHTWTHHLDBEN41 <> BLANK) OR (^CHTWTHHLDBEN42 <> NULL) OR (^CHTWTHHLDBEN43 <> NULL)) AND (((^CHTWTHHLDBEN39 = BLANK) AND (^CHTWTHHLDBEN40 = BLANK)) OR (^CHTWTHHLDBEN41 = BLANK) OR (^CHTWTHHLDBEN42 = NULL) OR (^CHTWTHHLDBEN43 = NULL))
    
            Data Elements:
    
            ^CHTWTHHLDBEN39 = Beneficiary.Residential:AddressDetails.Line1.Text
    
            ^CHTWTHHLDBEN40 = Beneficiary.Residential:AddressDetails.Line2.Text
    
            ^CHTWTHHLDBEN41 = Beneficiary.Residential:AddressDetails.LocalityName.Text
    
            ^CHTWTHHLDBEN42 = Beneficiary.Residential:AddressDetails.StateOrTerritory.Code
    
            ^CHTWTHHLDBEN43 = Beneficiary.Residential:AddressDetails.Postcode.Text
            */
            assertion = ((string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN39) != true || string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN40) != true || string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN41) != true || report.CHTWTHHLDBEN42 != null || report.CHTWTHHLDBEN43 != null) && (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN39) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN40) == true || string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN41) == true || report.CHTWTHHLDBEN42 == null || report.CHTWTHHLDBEN43 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000049", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary residential address details must be present",
                    LongDescription = @"Beneficiary residential address details must be complete with Street name and number, Suburb/town, State, and Postcode.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000049"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN39", Value = report.CHTWTHHLDBEN39 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN40", Value = report.CHTWTHHLDBEN40 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN41", Value = report.CHTWTHHLDBEN41 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN42", Value = report.CHTWTHHLDBEN42 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN43", Value = report.CHTWTHHLDBEN43 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000049
    
            #region VR.ATO.CHTWTHHLDBEN.000052
    
            /*  VR.ATO.CHTWTHHLDBEN.000052
            Address must be a physical location. A post office box, locked bag or private bag for example are not acceptable.
    
            Legacy Rule Format:
            IF ([chtwthhldben39] = FOUND("po box","gpo box","Locked Bag","Private Bag"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben39] = CHTWTHHLDBEN:Beneficiary.Residential:pyde.02.00:AddressDetails.Line1.Text

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN39, '"po box","gpo box","Locked Bag","Private Bag"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN39 = Beneficiary.Residential:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN39, @"(^(po box|gpo box|Locked Bag|Private Bag)$)|(^(.*)( po box | gpo box | Locked Bag | Private Bag )(.*)$)|(^(po box |gpo box |Locked Bag |Private Bag )(.*)$)|(^(.*)( po box| gpo box| Locked Bag| Private Bag)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000052", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary residential address must be a physical location",
                    LongDescription = @"Beneficiary residential address must be a physical location. A post office box, locked bag or private bag for example are not acceptable.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000052"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN39", Value = report.CHTWTHHLDBEN39 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000052
    
            #region VR.ATO.CHTWTHHLDBEN.000053
    
            /*  VR.ATO.CHTWTHHLDBEN.000053
            Address must be a physical location. A post office box, locked bag or private bag for example are not acceptable.
    
            Legacy Rule Format:
            IF ([chtwthhldben40] = FOUND("po box","gpo box","Locked Bag","Private Bag"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben40] = CHTWTHHLDBEN:Beneficiary.Residential:pyde.02.00:AddressDetails.Line2.Text

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN40, '"po box","gpo box","Locked Bag","Private Bag"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN40 = Beneficiary.Residential:AddressDetails.Line2.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN40, @"(^(po box|gpo box|Locked Bag|Private Bag)$)|(^(.*)( po box | gpo box | Locked Bag | Private Bag )(.*)$)|(^(po box |gpo box |Locked Bag |Private Bag )(.*)$)|(^(.*)( po box| gpo box| Locked Bag| Private Bag)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary residential address must be a physical location",
                    LongDescription = @"Beneficiary residential address must be a physical location. A post office box, locked bag or private bag for example are not acceptable.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000053"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN40", Value = report.CHTWTHHLDBEN40 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000053
    
            #region VR.ATO.CHTWTHHLDBEN.000055
    
            /*  VR.ATO.CHTWTHHLDBEN.000055
            Beneficiary Other given names has a maximum field length 30
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonNameDetails.OtherGivenName.Text) > 30
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN25) > 30
    
            Data Elements:
    
            ^CHTWTHHLDBEN25 = Beneficiary:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN25) > 30);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000055", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary other given names must not exceed 30 characters",
                    LongDescription = @"For the Beneficiary other given names, the maximum allowable characters for this form is 30",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000055"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN25", Value = report.CHTWTHHLDBEN25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000055

            #region VR.ATO.CHTWTHHLDBEN.000064

            /*  VR.ATO.CHTWTHHLDBEN.000064
            Address field contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben62] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.Line1.Text

            Technical Business Rule Format:
            (^CHTWTHHLDBEN62 <> BLANK) AND (StartsWithSet(^CHTWTHHLDBEN62,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^CHTWTHHLDBEN62, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN62 = Beneficiary.Postal:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN62) != true && IsMatch(report.CHTWTHHLDBEN62, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.CHTWTHHLDBEN62, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000064"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN62", Value = report.CHTWTHHLDBEN62 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000064
    
            #region VR.ATO.CHTWTHHLDBEN.000066
    
            /*  VR.ATO.CHTWTHHLDBEN.000066
            Address line 1 contains 'Care of' reference and no street address in Address line 2. If 'Care of' reference is on Address line 1, it must be followed by a name then Address line 2 must contain the street address.
    
            Legacy Rule Format:
            IF (([chtwthhldben62] CONTAINS "C/-") AND ([chtwthhldben63] = NULLORBLANK))
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben62] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.Line1.Text
            [chtwthhldben63] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.Line2.Text

            Technical Business Rule Format:
            ((Contains(^CHTWTHHLDBEN62, 'C/-')) AND (^CHTWTHHLDBEN63 = BLANK))
    
            Data Elements:
    
            ^CHTWTHHLDBEN62 = Beneficiary.Postal:AddressDetails.Line1.Text
    
            ^CHTWTHHLDBEN63 = Beneficiary.Postal:AddressDetails.Line2.Text
            */
            assertion = ((report.CHTWTHHLDBEN62 == null ? false : report.CHTWTHHLDBEN62.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN63) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428026", Severity = ProcessMessageSeverity.Error,
                    Description = @"C/- must be followed by a Name and the Address on line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000066"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN62", Value = report.CHTWTHHLDBEN62 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN63", Value = report.CHTWTHHLDBEN63 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000066
    
            #region VR.ATO.CHTWTHHLDBEN.000067
    
            /*  VR.ATO.CHTWTHHLDBEN.000067
            Numerics [0 to 9] are not allowed in the suburb/town field.
    
            Legacy Rule Format:
            IF ([chtwthhldben41] CONTAINS NUMERICS SET(0-9))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben41] = CHTWTHHLDBEN:Beneficiary.Residential:pyde.02.00:AddressDetails.LocalityName.Text

            Technical Business Rule Format:
            (ContainsSet(^CHTWTHHLDBEN41, '0-9'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN41 = Beneficiary.Residential:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN41, @"[0-9]",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000067"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN41", Value = report.CHTWTHHLDBEN41 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000067
    
            #region VR.ATO.CHTWTHHLDBEN.000068
    
            /*  VR.ATO.CHTWTHHLDBEN.000068
            Numerics [0 to 9] are not allowed in the suburb/town field.
    
            Legacy Rule Format:
            IF ([chtwthhldben69] CONTAINS NUMERICS SET(0-9))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben69] = CHTWTHHLDBEN:Beneficiary.Business:pyde.02.00:AddressDetails.LocalityName.Text

            Technical Business Rule Format:
            (ContainsSet(^CHTWTHHLDBEN69, '0-9'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN69 = Beneficiary.Business:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN69, @"[0-9]",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000068"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN69", Value = report.CHTWTHHLDBEN69 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000068
    
            #region VR.ATO.CHTWTHHLDBEN.000069
    
            /*  VR.ATO.CHTWTHHLDBEN.000069
            Numerics [0 to 9] are not allowed in the suburb/town field.
    
            Legacy Rule Format:
            IF (([chtwthhldben61] = NULL) OR ([chtwthhldben61] = "au")) AND ([chtwthhldben64] CONTAINS NUMERICS SET(0-9))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben61] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.08:AddressDetails.Country.Code
            [chtwthhldben64] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.LocalityName.Text

            Technical Business Rule Format:
            ((^CHTWTHHLDBEN61 = NULL) OR (^CHTWTHHLDBEN61 = 'au')) AND (ContainsSet(^CHTWTHHLDBEN64, '0-9'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN64 = Beneficiary.Postal:AddressDetails.LocalityName.Text
    
            ^CHTWTHHLDBEN61 = Beneficiary.Postal:AddressDetails.Country.Code
            */
            assertion = ((report.CHTWTHHLDBEN61 == null || report.CHTWTHHLDBEN61 == @"au") && IsMatch(report.CHTWTHHLDBEN64, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000069"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN61", Value = report.CHTWTHHLDBEN61 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN64", Value = report.CHTWTHHLDBEN64 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000069
    
            #region VR.ATO.CHTWTHHLDBEN.000070
    
            /*  VR.ATO.CHTWTHHLDBEN.000070
            State code [QLD,NSW,VIC,SA,WA,NT,ACT,TAS] or a leading space or combination of spaces with a State code is present in the suburb/town field.
    
            Legacy Rule Format:
            IF ([chtwthhldben41] = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben41] = CHTWTHHLDBEN:Beneficiary.Residential:pyde.02.00:AddressDetails.LocalityName.Text

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN41, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN41 = Beneficiary.Residential:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN41, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000070"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN41", Value = report.CHTWTHHLDBEN41 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000070
    
            #region VR.ATO.CHTWTHHLDBEN.000071
    
            /*  VR.ATO.CHTWTHHLDBEN.000071
            State code [QLD,NSW,VIC,SA,WA,NT,ACT,TAS] or a leading space or combination of spaces with a State code is present in the suburb/town field.
    
            Legacy Rule Format:
            IF ([chtwthhldben69] = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben69] = CHTWTHHLDBEN:Beneficiary.Business:pyde.02.00:AddressDetails.LocalityName.Text

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN69, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN69 = Beneficiary.Business:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN69, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000071"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN69", Value = report.CHTWTHHLDBEN69 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000071
    
            #region VR.ATO.CHTWTHHLDBEN.000072
    
            /*  VR.ATO.CHTWTHHLDBEN.000072
            State code [QLD,NSW,VIC,SA,WA,NT,ACT,TAS] or a leading space or combination of spaces with a State code is present in the suburb/town field.
    
            Legacy Rule Format:
            IF (([chtwthhldben61] = NULL) OR ([chtwthhldben61] = "au")) AND ([chtwthhldben64] = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben61] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.08:AddressDetails.Country.Code
            [chtwthhldben64] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.LocalityName.Text

            Technical Business Rule Format:
            ((^CHTWTHHLDBEN61 = NULL) OR (^CHTWTHHLDBEN61 = 'au')) AND (FoundSet(^CHTWTHHLDBEN64, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN64 = Beneficiary.Postal:AddressDetails.LocalityName.Text
    
            ^CHTWTHHLDBEN61 = Beneficiary.Postal:AddressDetails.Country.Code
            */
            assertion = ((report.CHTWTHHLDBEN61 == null || report.CHTWTHHLDBEN61 == @"au") && IsMatch(report.CHTWTHHLDBEN64, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000072"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN61", Value = report.CHTWTHHLDBEN61 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN64", Value = report.CHTWTHHLDBEN64 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000072
    
            #region VR.ATO.CHTWTHHLDBEN.000073
    
            /*  VR.ATO.CHTWTHHLDBEN.000073
            Country name is present with State or postcode in the address field.  Determine whether the address is Australian or overseas and correct accordingly.
    
            Legacy Rule Format:
            IF ([chtwthhldben61] <> NULL) AND ([chtwthhldben61] <> "au") AND (([chtwthhldben65] <> NULL) OR ([chtwthhldben66] <> NULL))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben61] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.08:AddressDetails.Country.Code
            [chtwthhldben65] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.StateOrTerritory.Code
            [chtwthhldben66] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.Postcode.Text

            Technical Business Rule Format:
            (^CHTWTHHLDBEN61 <> NULL) AND (^CHTWTHHLDBEN61 <> 'au') AND ((^CHTWTHHLDBEN65 <> NULL) OR (^CHTWTHHLDBEN66 <> NULL))
    
            Data Elements:
    
            ^CHTWTHHLDBEN61 = Beneficiary.Postal:AddressDetails.Country.Code
    
            ^CHTWTHHLDBEN65 = Beneficiary.Postal:AddressDetails.StateOrTerritory.Code
    
            ^CHTWTHHLDBEN66 = Beneficiary.Postal:AddressDetails.Postcode.Text
            */
            assertion = (report.CHTWTHHLDBEN61 != null && report.CHTWTHHLDBEN61 != @"au" && (report.CHTWTHHLDBEN65 != null || report.CHTWTHHLDBEN66 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000073", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and postcode must not be present for non-Australian addresses",
                    LongDescription = @"Country name is present with State or postcode in the Beneficiary postal address field.  Determine whether the address is Australian or overseas and correct accordingly.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Country.Code[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000073"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN61", Value = report.CHTWTHHLDBEN61 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN65", Value = report.CHTWTHHLDBEN65 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN66", Value = report.CHTWTHHLDBEN66 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000073
    
            #region VR.ATO.CHTWTHHLDBEN.000074
    
            /*  VR.ATO.CHTWTHHLDBEN.000074
            State is present in the address field without a postcode.
    
            Legacy Rule Format:
            IF ([chtwthhldben65] <> NULL) AND ([chtwthhldben66] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben65] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.StateOrTerritory.Code
            [chtwthhldben66] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.Postcode.Text

            Technical Business Rule Format:
            (^CHTWTHHLDBEN65 <> NULL) AND (^CHTWTHHLDBEN66 = NULL)
    
            Data Elements:
    
            ^CHTWTHHLDBEN65 = Beneficiary.Postal:AddressDetails.StateOrTerritory.Code
    
            ^CHTWTHHLDBEN66 = Beneficiary.Postal:AddressDetails.Postcode.Text
            */
            assertion = (report.CHTWTHHLDBEN65 != null && report.CHTWTHHLDBEN66 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410055", Severity = ProcessMessageSeverity.Error,
                    Description = @"Postcode must be present if State is present",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000074"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN65", Value = report.CHTWTHHLDBEN65 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN66", Value = report.CHTWTHHLDBEN66 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000074
    
            #region VR.ATO.CHTWTHHLDBEN.000075
    
            /*  VR.ATO.CHTWTHHLDBEN.000075
            Postcode is present in the address field without a State code.
    
            Legacy Rule Format:
            IF ([chtwthhldben66] <> NULL) AND ([chtwthhldben65] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben65] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.StateOrTerritory.Code
            [chtwthhldben66] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.Postcode.Text

            Technical Business Rule Format:
            (^CHTWTHHLDBEN66 <> NULL) AND (^CHTWTHHLDBEN65 = NULL)
    
            Data Elements:
    
            ^CHTWTHHLDBEN65 = Beneficiary.Postal:AddressDetails.StateOrTerritory.Code
    
            ^CHTWTHHLDBEN66 = Beneficiary.Postal:AddressDetails.Postcode.Text
            */
            assertion = (report.CHTWTHHLDBEN66 != null && report.CHTWTHHLDBEN65 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410017", Severity = ProcessMessageSeverity.Error,
                    Description = @"State must be present if Postcode is present",
                    LongDescription = @"If the address is an Australian address, then state and postcode must be present. If the address is an overseas address, then country must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000075"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN66", Value = report.CHTWTHHLDBEN66 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN65", Value = report.CHTWTHHLDBEN65 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000075
    
            #region VR.ATO.CHTWTHHLDBEN.000076
    
            /*  VR.ATO.CHTWTHHLDBEN.000076
            State, postcode, or country are missing from the address field.  Determine whether the address is Australian or overseas and correct accordingly.
    
            Legacy Rule Format:
            IF ([chtwthhldben65] = NULL) AND ([chtwthhldben66] = NULL) AND (([chtwthhldben61] = NULL) OR ([chtwthhldben61] = "au"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben61] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.08:AddressDetails.Country.Code
            [chtwthhldben65] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.StateOrTerritory.Code
            [chtwthhldben66] = CHTWTHHLDBEN:Beneficiary.Postal:pyde.02.00:AddressDetails.Postcode.Text

            Technical Business Rule Format:
            (^CHTWTHHLDBEN65 = NULL) AND (^CHTWTHHLDBEN66 = NULL) AND ((^CHTWTHHLDBEN61 = NULL) OR (^CHTWTHHLDBEN61 = 'au'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN65 = Beneficiary.Postal:AddressDetails.StateOrTerritory.Code
    
            ^CHTWTHHLDBEN61 = Beneficiary.Postal:AddressDetails.Country.Code
    
            ^CHTWTHHLDBEN66 = Beneficiary.Postal:AddressDetails.Postcode.Text
            */
            assertion = (report.CHTWTHHLDBEN65 == null && report.CHTWTHHLDBEN66 == null && (report.CHTWTHHLDBEN61 == null || report.CHTWTHHLDBEN61 == @"au"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000076", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and postcode are required for Australian addresses",
                    LongDescription = @"State, postcode, or country are required in the Beneficiary postal address field.  Determine whether the address is Australian or overseas and correct accordingly.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000076"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN65", Value = report.CHTWTHHLDBEN65 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN66", Value = report.CHTWTHHLDBEN66 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN61", Value = report.CHTWTHHLDBEN61 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000076
    
            #region VR.ATO.CHTWTHHLDBEN.000081
    
            /*  VR.ATO.CHTWTHHLDBEN.000081
            Address details must be complete with street name and number, suburb/town, State, and postcode.
    
            Legacy Rule Format:
            IF (([chtwthhldben60] <> NULLORBLANK) OR ([chtwthhldben59] <> NULLORBLANK) OR ([chtwthhldben69] <> NULLORBLANK) OR ([chtwthhldben70] <> NULL) OR ([chtwthhldben71] <> NULL))
            AND ((([chtwthhldben60] = NULLORBLANK) AND ([chtwthhldben59] = NULLORBLANK)) OR ([chtwthhldben69] = NULLORBLANK) OR ([chtwthhldben70] = NULL) OR ([chtwthhldben71] = NULL))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben59] = CHTWTHHLDBEN:Beneficiary.Business:pyde.02.00:AddressDetails.Line2.Text
            [chtwthhldben60] = CHTWTHHLDBEN:Beneficiary.Business:pyde.02.00:AddressDetails.Line1.Text
            [chtwthhldben69] = CHTWTHHLDBEN:Beneficiary.Business:pyde.02.00:AddressDetails.LocalityName.Text
            [chtwthhldben70] = CHTWTHHLDBEN:Beneficiary.Business:pyde.02.00:AddressDetails.StateOrTerritory.Code
            [chtwthhldben71] = CHTWTHHLDBEN:Beneficiary.Business:pyde.02.00:AddressDetails.Postcode.Text

            Technical Business Rule Format:
            ((^CHTWTHHLDBEN60 <> BLANK) OR (^CHTWTHHLDBEN59 <> BLANK) OR (^CHTWTHHLDBEN69 <> BLANK) OR (^CHTWTHHLDBEN70 <> NULL) OR (^CHTWTHHLDBEN71 <> NULL)) AND (((^CHTWTHHLDBEN60 = BLANK) AND (^CHTWTHHLDBEN59 = BLANK)) OR (^CHTWTHHLDBEN69 = BLANK) OR (^CHTWTHHLDBEN70 = NULL) OR (^CHTWTHHLDBEN71 = NULL))
    
            Data Elements:
    
            ^CHTWTHHLDBEN60 = Beneficiary.Business:AddressDetails.Line1.Text
    
            ^CHTWTHHLDBEN59 = Beneficiary.Business:AddressDetails.Line2.Text
    
            ^CHTWTHHLDBEN69 = Beneficiary.Business:AddressDetails.LocalityName.Text
    
            ^CHTWTHHLDBEN70 = Beneficiary.Business:AddressDetails.StateOrTerritory.Code
    
            ^CHTWTHHLDBEN71 = Beneficiary.Business:AddressDetails.Postcode.Text
            */
            assertion = ((string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN60) != true || string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN59) != true || string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN69) != true || report.CHTWTHHLDBEN70 != null || report.CHTWTHHLDBEN71 != null) && (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN60) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN59) == true || string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN69) == true || report.CHTWTHHLDBEN70 == null || report.CHTWTHHLDBEN71 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000081", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary business address details must be present",
                    LongDescription = @"Beneficiary business address details must be complete with street name and number, suburb/town, State, and postcode.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000081"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN60", Value = report.CHTWTHHLDBEN60 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN59", Value = report.CHTWTHHLDBEN59 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN69", Value = report.CHTWTHHLDBEN69 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN70", Value = report.CHTWTHHLDBEN70 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN71", Value = report.CHTWTHHLDBEN71 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000081
    
            #region VR.ATO.CHTWTHHLDBEN.000082
    
            /*  VR.ATO.CHTWTHHLDBEN.000082
            Address must be a physical location. A post office box, locked bag or private bag for example are not acceptable.
    
            Legacy Rule Format:
            IF ([chtwthhldben60] = FOUND("po box","gpo box","Locked Bag","Private Bag"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben60] = CHTWTHHLDBEN:Beneficiary.Business:pyde.02.00:AddressDetails.Line1.Text

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN60, '"po box","gpo box","Locked Bag","Private Bag"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN60 = Beneficiary.Business:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN60, @"(^(po box|gpo box|Locked Bag|Private Bag)$)|(^(.*)( po box | gpo box | Locked Bag | Private Bag )(.*)$)|(^(po box |gpo box |Locked Bag |Private Bag )(.*)$)|(^(.*)( po box| gpo box| Locked Bag| Private Bag)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000082", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary business address must be a physical location",
                    LongDescription = @"Beneficiary business address must be a physical location. A post office box, locked bag or private bag for example are not acceptable.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000082"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN60", Value = report.CHTWTHHLDBEN60 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000082
    
            #region VR.ATO.CHTWTHHLDBEN.000083
    
            /*  VR.ATO.CHTWTHHLDBEN.000083
            Address must be a physical location. A post office box, locked bag or private bag for example are not acceptable.
    
            Legacy Rule Format:
            IF ([chtwthhldben59] = FOUND("po box","gpo box","Locked Bag","Private Bag"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben59] = CHTWTHHLDBEN:Beneficiary.Business:pyde.02.00:AddressDetails.Line2.Text

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN59, '"po box","gpo box","Locked Bag","Private Bag"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN59 = Beneficiary.Business:AddressDetails.Line2.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN59, @"(^(po box|gpo box|Locked Bag|Private Bag)$)|(^(.*)( po box | gpo box | Locked Bag | Private Bag )(.*)$)|(^(po box |gpo box |Locked Bag |Private Bag )(.*)$)|(^(.*)( po box| gpo box| Locked Bag| Private Bag)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000083", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary business address must be a physical location",
                    LongDescription = @"Beneficiary business address must be a physical location. A post office box, locked bag or private bag for example are not acceptable.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000083"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN59", Value = report.CHTWTHHLDBEN59 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000083
    
            #region VR.ATO.CHTWTHHLDBEN.000091
    
            /*  VR.ATO.CHTWTHHLDBEN.000091
            Country code must be blank or set to Australia
    
            Legacy Rule Format:
            IF ([chtwthhldben29] <> NULL) AND ([chtwthhldben29] <> "au")
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben29] = CHTWTHHLDBEN:Beneficiary.Residential:pyde.02.08:AddressDetails.Country.Code

            Technical Business Rule Format:
            (^CHTWTHHLDBEN29 <> NULL) AND (^CHTWTHHLDBEN29 <> 'au')
    
            Data Elements:
    
            ^CHTWTHHLDBEN29 = Beneficiary.Residential:AddressDetails.Country.Code
            */
            assertion = (report.CHTWTHHLDBEN29 != null && report.CHTWTHHLDBEN29 != @"au");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000480", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be blank or set to Australia",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Country.Code[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000091"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN29", Value = report.CHTWTHHLDBEN29 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000091
    
            #region VR.ATO.CHTWTHHLDBEN.000092
    
            /*  VR.ATO.CHTWTHHLDBEN.000092
            Country code must be blank or set to Australia
    
            Legacy Rule Format:
            IF ([chtwthhldben30] <> NULL) AND ([chtwthhldben30] <> "au")
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhldben30] = CHTWTHHLDBEN:Beneficiary.Business:pyde.02.08:AddressDetails.Country.Code

            Technical Business Rule Format:
            (^CHTWTHHLDBEN30 <> NULL) AND (^CHTWTHHLDBEN30 <> 'au')
    
            Data Elements:
    
            ^CHTWTHHLDBEN30 = Beneficiary.Business:AddressDetails.Country.Code
            */
            assertion = (report.CHTWTHHLDBEN30 != null && report.CHTWTHHLDBEN30 != @"au");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000480", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be blank or set to Australia",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Country.Code[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000092"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN30", Value = report.CHTWTHHLDBEN30 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000092
    
            #region VR.ATO.CHTWTHHLDBEN.000097
    
            /*  VR.ATO.CHTWTHHLDBEN.000097
            Beneficiary Amount withheld has a maximum field length 12
    
            Legacy Rule Format:
            IF rvctc2.xx.xx:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount <> MONETARY(U,12,0)
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CHTWTHHLDBEN45, 'U', 12, 0)
    
            Data Elements:
    
            ^CHTWTHHLDBEN45 = Beneficiary:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount
            */
            assertion = NotMonetary(report.CHTWTHHLDBEN45, @"U", 12, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000097", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary amount withheld is not in a valid monetary format",
                    LongDescription = @"Beneficiary amount withheld must be unsigned, not exceed 12 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000097"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN45", Value = report.CHTWTHHLDBEN45.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000097
    
            #region VR.ATO.CHTWTHHLDBEN.000098
    
            /*  VR.ATO.CHTWTHHLDBEN.000098
            Beneficiary Amount of payments from which withholdings were made has a maximum field length 12
    
            Legacy Rule Format:
            IF rvctc2.xx.xx:IncomeTax.PayAsYouGoWithholding.PaymentGross.Amount <> MONETARY(U,12,0)
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CHTWTHHLDBEN46, 'U', 12, 0)
    
            Data Elements:
    
            ^CHTWTHHLDBEN46 = Beneficiary:IncomeTax.PayAsYouGoWithholding.PaymentGross.Amount
            */
            assertion = NotMonetary(report.CHTWTHHLDBEN46, @"U", 12, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000098", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary amount of payments is not in a valid monetary format",
                    LongDescription = @"Beneficiary amount of payments must be unsigned, not exceed 12 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.PaymentGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000098"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN46", Value = report.CHTWTHHLDBEN46.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000098
    
            #region VR.ATO.CHTWTHHLDBEN.000202
    
            /*  VR.ATO.CHTWTHHLDBEN.000202
            Ensure that only a single beneficiary entity is covered within a single instance
    
            Legacy Rule Format:
            IF Beneficiary:entity.identifier.ID <> ANY OTHER OCCURRENCE OF entity.identifier.ID
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Identifier1 <> ^Identifier
    
            Data Elements:
    
            ^Identifier = Beneficiary
    
            ^Identifier1 = Beneficiary:Beneficiary.Business
            */
            assertion = ((report.BeneficiaryBusinessExists == true) && (report.BeneficiaryBusinessIdentifier != report.BeneficiaryIdentifier));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000202", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Beneficiary entity id's mismatch.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000202"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000202
    
            #region VR.ATO.CHTWTHHLDBEN.000202
    
            /*  VR.ATO.CHTWTHHLDBEN.000202
            Ensure that only a single beneficiary entity is covered within a single instance
    
            Legacy Rule Format:
            IF Beneficiary:entity.identifier.ID <> ANY OTHER OCCURRENCE OF entity.identifier.ID
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Identifier1 <> ^Identifier
    
            Data Elements:
    
            ^Identifier = Beneficiary
    
            ^Identifier1 = Beneficiary:Beneficiary.Postal
            */
            assertion = ((report.BeneficiaryPostalExists == true) && (report.BeneficiaryPostalIdentifier != report.BeneficiaryIdentifier));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000202", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Beneficiary entity id's mismatch.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000202"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000202
    
            #region VR.ATO.CHTWTHHLDBEN.000202
    
            /*  VR.ATO.CHTWTHHLDBEN.000202
            Ensure that only a single beneficiary entity is covered within a single instance
    
            Legacy Rule Format:
            IF Beneficiary:entity.identifier.ID <> ANY OTHER OCCURRENCE OF entity.identifier.ID
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Identifier1 <> ^Identifier
    
            Data Elements:
    
            ^Identifier = Beneficiary
    
            ^Identifier1 = Beneficiary:Beneficiary.Residential
            */
            assertion = ((report.BeneficiaryResidentialExists == true) && (report.BeneficiaryResidentialIdentifier != report.BeneficiaryIdentifier));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000202", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Beneficiary entity id's mismatch.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000202"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000202
    
            #region VR.ATO.CHTWTHHLDBEN.000203
    
            /*  VR.ATO.CHTWTHHLDBEN.000203
            Beneficiary Postal address line 1 has a maximum field length 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN60) > 38
    
            Data Elements:
    
            ^CHTWTHHLDBEN60 = Beneficiary.Business:AddressDetails.Line1.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN60) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000203", Severity = ProcessMessageSeverity.Error,
                    Description = @"Each Beneficiary street name and number field must not exceed 38 characters",
                    LongDescription = @"For each Beneficiary street name and number field, the maximum allowable characters for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000203"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN60", Value = report.CHTWTHHLDBEN60 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000203
    
            #region VR.ATO.CHTWTHHLDBEN.000203
    
            /*  VR.ATO.CHTWTHHLDBEN.000203
            Beneficiary Postal address line 1 has a maximum field length 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN62) > 38
    
            Data Elements:
    
            ^CHTWTHHLDBEN62 = Beneficiary.Postal:AddressDetails.Line1.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN62) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000203", Severity = ProcessMessageSeverity.Error,
                    Description = @"Each Beneficiary street name and number field must not exceed 38 characters",
                    LongDescription = @"For each Beneficiary street name and number field, the maximum allowable characters for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000203"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN62", Value = report.CHTWTHHLDBEN62 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000203
    
            #region VR.ATO.CHTWTHHLDBEN.000203
    
            /*  VR.ATO.CHTWTHHLDBEN.000203
            Beneficiary Postal address line 1 has a maximum field length 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN39) > 38
    
            Data Elements:
    
            ^CHTWTHHLDBEN39 = Beneficiary.Residential:AddressDetails.Line1.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN39) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000203", Severity = ProcessMessageSeverity.Error,
                    Description = @"Each Beneficiary street name and number field must not exceed 38 characters",
                    LongDescription = @"For each Beneficiary street name and number field, the maximum allowable characters for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000203"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN39", Value = report.CHTWTHHLDBEN39 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000203
    
            #region VR.ATO.CHTWTHHLDBEN.000205
    
            /*  VR.ATO.CHTWTHHLDBEN.000205
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS SET("{","}","|","_","\","#"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^CHTWTHHLDBEN23, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN23, @"\{|\}|\||_|\\|#",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000205", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary family name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000205"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000205
    
            #region VR.ATO.CHTWTHHLDBEN.000206
    
            /*  VR.ATO.CHTWTHHLDBEN.000206
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS SET("{","}","|","_","\","#"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^CHTWTHHLDBEN24, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^CHTWTHHLDBEN24 = Beneficiary:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN24, @"\{|\}|\||_|\\|#",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000206", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary first name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000206"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN24", Value = report.CHTWTHHLDBEN24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000206
    
            #region VR.ATO.CHTWTHHLDBEN.000207
    
            /*  VR.ATO.CHTWTHHLDBEN.000207
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS SET("{","}","|","_","\","#"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^CHTWTHHLDBEN25, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^CHTWTHHLDBEN25 = Beneficiary:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN25, @"\{|\}|\||_|\\|#",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000207", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary other given names contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000207"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN25", Value = report.CHTWTHHLDBEN25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000207
    
            #region VR.ATO.CHTWTHHLDBEN.000208
    
            /*  VR.ATO.CHTWTHHLDBEN.000208
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? /# or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("{","}","|","_","\"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^CHTWTHHLDBEN32, '"{","}","|","_","\"')
    
            Data Elements:
    
            ^CHTWTHHLDBEN32 = Beneficiary:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN32, @"\{|\}|\||_|\\",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000208", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary non-individual name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000208"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN32", Value = report.CHTWTHHLDBEN32 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000208
    
            #region VR.ATO.CHTWTHHLDBEN.000209
    
            /*  VR.ATO.CHTWTHHLDBEN.000209
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:AddressDetails.Line1.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN60 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLDBEN60, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN60 = Beneficiary.Business:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN60) != true && !(IsMatch(report.CHTWTHHLDBEN60, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000209", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary street name and number contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000209"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN60", Value = report.CHTWTHHLDBEN60 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000209
    
            #region VR.ATO.CHTWTHHLDBEN.000209
    
            /*  VR.ATO.CHTWTHHLDBEN.000209
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:AddressDetails.Line1.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN62 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLDBEN62, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN62 = Beneficiary.Postal:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN62) != true && !(IsMatch(report.CHTWTHHLDBEN62, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000209", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary street name and number contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000209"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN62", Value = report.CHTWTHHLDBEN62 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000209
    
            #region VR.ATO.CHTWTHHLDBEN.000209
    
            /*  VR.ATO.CHTWTHHLDBEN.000209
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:AddressDetails.Line1.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN39 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLDBEN39, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN39 = Beneficiary.Residential:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN39) != true && !(IsMatch(report.CHTWTHHLDBEN39, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000209", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary street name and number contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000209"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN39", Value = report.CHTWTHHLDBEN39 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000209
    
            #region VR.ATO.CHTWTHHLDBEN.000210
    
            /*  VR.ATO.CHTWTHHLDBEN.000210
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:AddressDetails.Line2.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN59 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLDBEN59, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN59 = Beneficiary.Business:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN59) != true && !(IsMatch(report.CHTWTHHLDBEN59, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000210", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary street name and number contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000210"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN59", Value = report.CHTWTHHLDBEN59 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000210
    
            #region VR.ATO.CHTWTHHLDBEN.000210
    
            /*  VR.ATO.CHTWTHHLDBEN.000210
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:AddressDetails.Line2.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN63 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLDBEN63, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN63 = Beneficiary.Postal:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN63) != true && !(IsMatch(report.CHTWTHHLDBEN63, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000210", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary street name and number contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000210"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN63", Value = report.CHTWTHHLDBEN63 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000210
    
            #region VR.ATO.CHTWTHHLDBEN.000210
    
            /*  VR.ATO.CHTWTHHLDBEN.000210
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:AddressDetails.Line2.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN40 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLDBEN40, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN40 = Beneficiary.Residential:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN40) != true && !(IsMatch(report.CHTWTHHLDBEN40, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000210", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary street name and number contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000210"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN40", Value = report.CHTWTHHLDBEN40 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000210
    
            #region VR.ATO.CHTWTHHLDBEN.000211
    
            /*  VR.ATO.CHTWTHHLDBEN.000211
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:AddressDetails.LocalityName.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN69 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLDBEN69, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN69 = Beneficiary.Business:AddressDetails.LocalityName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN69) != true && !(IsMatch(report.CHTWTHHLDBEN69, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary suburb/town contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000211"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN69", Value = report.CHTWTHHLDBEN69 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000211
    
            #region VR.ATO.CHTWTHHLDBEN.000211
    
            /*  VR.ATO.CHTWTHHLDBEN.000211
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:AddressDetails.LocalityName.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN64 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLDBEN64, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN64 = Beneficiary.Postal:AddressDetails.LocalityName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN64) != true && !(IsMatch(report.CHTWTHHLDBEN64, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary suburb/town contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000211"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN64", Value = report.CHTWTHHLDBEN64 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000211
    
            #region VR.ATO.CHTWTHHLDBEN.000211
    
            /*  VR.ATO.CHTWTHHLDBEN.000211
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:AddressDetails.LocalityName.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN41 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLDBEN41, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN41 = Beneficiary.Residential:AddressDetails.LocalityName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN41) != true && !(IsMatch(report.CHTWTHHLDBEN41, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary suburb/town contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000211"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN41", Value = report.CHTWTHHLDBEN41 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000211
    
            #region VR.ATO.CHTWTHHLDBEN.000218
    
            /*  VR.ATO.CHTWTHHLDBEN.000218
            Beneficiary Postal address line 2 has a maximum field length 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN59) > 38
    
            Data Elements:
    
            ^CHTWTHHLDBEN59 = Beneficiary.Business:AddressDetails.Line2.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN59) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000218", Severity = ProcessMessageSeverity.Error,
                    Description = @"Each Beneficiary street name and number field must not exceed 38 characters",
                    LongDescription = @"For each Beneficiary street name and number field, the maximum allowable characters for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000218"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN59", Value = report.CHTWTHHLDBEN59 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000218
    
            #region VR.ATO.CHTWTHHLDBEN.000218
    
            /*  VR.ATO.CHTWTHHLDBEN.000218
            Beneficiary Postal address line 2 has a maximum field length 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN63) > 38
    
            Data Elements:
    
            ^CHTWTHHLDBEN63 = Beneficiary.Postal:AddressDetails.Line2.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN63) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000218", Severity = ProcessMessageSeverity.Error,
                    Description = @"Each Beneficiary street name and number field must not exceed 38 characters",
                    LongDescription = @"For each Beneficiary street name and number field, the maximum allowable characters for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000218"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN63", Value = report.CHTWTHHLDBEN63 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000218
    
            #region VR.ATO.CHTWTHHLDBEN.000218
    
            /*  VR.ATO.CHTWTHHLDBEN.000218
            Beneficiary Postal address line 2 has a maximum field length 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLDBEN40) > 38
    
            Data Elements:
    
            ^CHTWTHHLDBEN40 = Beneficiary.Residential:AddressDetails.Line2.Text
            */
            assertion = (Length(report.CHTWTHHLDBEN40) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000218", Severity = ProcessMessageSeverity.Error,
                    Description = @"Each Beneficiary street name and number field must not exceed 38 characters",
                    LongDescription = @"For each Beneficiary street name and number field, the maximum allowable characters for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000218"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN40", Value = report.CHTWTHHLDBEN40 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000218
    
            #region VR.ATO.CHTWTHHLDBEN.000219
    
            /*  VR.ATO.CHTWTHHLDBEN.000219
            The TFN has failed the algorithm check.
    
            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.TaxFileNumber.Identifier <> NULLORBLANK) AND (TFNALGORITHM (pyid.xx.xx:Identifiers.TaxFileNumber.Identifier) = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN52 <> BLANK) AND (FailsTFNAlgorithm(^CHTWTHHLDBEN52))
    
            Data Elements:
    
            ^CHTWTHHLDBEN52 = Beneficiary:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN52) != true && FailsTFNAlgorithm(report.CHTWTHHLDBEN52));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000219", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary Tax File Number has failed the algorithm check",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxFileNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000219"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN52", Value = report.CHTWTHHLDBEN52 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000219
    
            #region VR.ATO.CHTWTHHLDBEN.000220
    
            /*  VR.ATO.CHTWTHHLDBEN.000220
            Scheme for Beneficiary must be set to http://www.sbr.gov.au/id
    
            Legacy Rule Format:
            IF (Beneficiary:entity.identifier.scheme <> "http://www.sbr.gov.au/id")
                RETURN VALIDATION MESSAGE
             ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.sbr.gov.au/id'
    
            Data Elements:
    
            ^IdentifierScheme = Beneficiary
            */
            assertion = ((report.BeneficiaryExists == true) && (report.BeneficiaryIdentifierScheme != @"http://www.sbr.gov.au/id"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000220", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Beneficiary must be ""http://www.sbr.gov.au/id""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000220"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000220
    
            #region VR.ATO.CHTWTHHLDBEN.000221
    
            /*  VR.ATO.CHTWTHHLDBEN.000221
            Scheme for Beneficiary.Residential must be set to http://www.sbr.gov.au/id
    
            Legacy Rule Format:
            IF (Beneficiary.Residential:entity.identifier.scheme <> "http://www.sbr.gov.au/id")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.sbr.gov.au/id'
    
            Data Elements:
    
            ^IdentifierScheme = Beneficiary:Beneficiary.Residential
            */
            assertion = ((report.BeneficiaryResidentialExists == true) && (report.BeneficiaryResidentialIdentifierScheme != @"http://www.sbr.gov.au/id"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000221", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Beneficiary.Residential must be ""http://www.sbr.gov.au/id""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryResidentialOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000221"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000221
    
            #region VR.ATO.CHTWTHHLDBEN.000222
    
            /*  VR.ATO.CHTWTHHLDBEN.000222
            Scheme for Beneficiary.Business must be set to http://www.sbr.gov.au/id
    
            Legacy Rule Format:
            IF (Beneficiary.Business:entity.identifier.scheme <> "http://www.sbr.gov.au/id")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.sbr.gov.au/id'
    
            Data Elements:
    
            ^IdentifierScheme = Beneficiary:Beneficiary.Business
            */
            assertion = ((report.BeneficiaryBusinessExists == true) && (report.BeneficiaryBusinessIdentifierScheme != @"http://www.sbr.gov.au/id"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000222", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Beneficiary.Business must be ""http://www.sbr.gov.au/id""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryBusinessOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000222"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000222
    
            #region VR.ATO.CHTWTHHLDBEN.000223
    
            /*  VR.ATO.CHTWTHHLDBEN.000223
            Scheme for Beneficiary.Postal must be set to http://www.sbr.gov.au/id
    
            Legacy Rule Format:
            IF (Beneficiary.Postal:entity.identifier.scheme <> "http://www.sbr.gov.au/id")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.sbr.gov.au/id'
    
            Data Elements:
    
            ^IdentifierScheme = Beneficiary:Beneficiary.Postal
            */
            assertion = ((report.BeneficiaryPostalExists == true) && (report.BeneficiaryPostalIdentifierScheme != @"http://www.sbr.gov.au/id"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000223", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Beneficiary.Postal must be ""http://www.sbr.gov.au/id""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryPostalOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000223"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000223
    
            #region VR.ATO.CHTWTHHLDBEN.000224
    
            /*  VR.ATO.CHTWTHHLDBEN.000224
            Invalid context. The number of Beneficiary contexts must equal 1.
    
            Legacy Rule Format:
            IF COUNT(Beneficiary) <> 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = Beneficiary
            */
            assertion = (Count(report.BeneficiaryCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000224", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Beneficiary contexts must equal 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000224"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000224
    
            #region VR.ATO.CHTWTHHLDBEN.000227
    
            /*  VR.ATO.CHTWTHHLDBEN.000227
            Invalid context. The number of Beneficiary.Postal contexts must equal 1.
    
            Legacy Rule Format:
            IF COUNT(Beneficiary.Postal) <> 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = Beneficiary:Beneficiary.Postal
            */
            assertion = (Count(report.BeneficiaryPostalCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLDBEN.000227", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Beneficiary.Postal contexts must equal 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryPostalLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000227"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000227
    
            #region VR.ATO.CHTWTHHLDBEN.000228
    
            /*  VR.ATO.CHTWTHHLDBEN.000228
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.startDate WHERE CONTEXT(ALL) <> Beneficiary:period.startDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate1 <> NULL AND ^StartDate1 <> ^StartDate
    
            Data Elements:
    
            ^StartDate = Beneficiary
    
            ^StartDate1 = Beneficiary:Beneficiary.Business
            */
            assertion = ((report.BeneficiaryBusinessExists == true) && (report.BeneficiaryBusinessStartDate != null && report.BeneficiaryBusinessStartDate != report.BeneficiaryStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000228"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000228
    
            #region VR.ATO.CHTWTHHLDBEN.000228
    
            /*  VR.ATO.CHTWTHHLDBEN.000228
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.startDate WHERE CONTEXT(ALL) <> Beneficiary:period.startDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate1 <> NULL AND ^StartDate1 <> ^StartDate
    
            Data Elements:
    
            ^StartDate = Beneficiary
    
            ^StartDate1 = Beneficiary:Beneficiary.Postal
            */
            assertion = ((report.BeneficiaryPostalExists == true) && (report.BeneficiaryPostalStartDate != null && report.BeneficiaryPostalStartDate != report.BeneficiaryStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000228"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000228
    
            #region VR.ATO.CHTWTHHLDBEN.000228
    
            /*  VR.ATO.CHTWTHHLDBEN.000228
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.startDate WHERE CONTEXT(ALL) <> Beneficiary:period.startDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate1 <> NULL AND ^StartDate1 <> ^StartDate
    
            Data Elements:
    
            ^StartDate = Beneficiary
    
            ^StartDate1 = Beneficiary:Beneficiary.Residential
            */
            assertion = ((report.BeneficiaryResidentialExists == true) && (report.BeneficiaryResidentialStartDate != null && report.BeneficiaryResidentialStartDate != report.BeneficiaryStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000228"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000228
    
            #region VR.ATO.CHTWTHHLDBEN.000229
    
            /*  VR.ATO.CHTWTHHLDBEN.000229
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.endDate WHERE CONTEXT(ALL) <> Beneficiary:period.endDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^EndDate1 <> NULL AND ^EndDate1 <> ^EndDate
    
            Data Elements:
    
            ^EndDate = Beneficiary
    
            ^EndDate1 = Beneficiary:Beneficiary.Business
            */
            assertion = ((report.BeneficiaryBusinessExists == true) && (report.BeneficiaryBusinessEndDate != null && report.BeneficiaryBusinessEndDate != report.BeneficiaryEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000229"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000229
    
            #region VR.ATO.CHTWTHHLDBEN.000229
    
            /*  VR.ATO.CHTWTHHLDBEN.000229
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.endDate WHERE CONTEXT(ALL) <> Beneficiary:period.endDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^EndDate1 <> NULL AND ^EndDate1 <> ^EndDate
    
            Data Elements:
    
            ^EndDate = Beneficiary
    
            ^EndDate1 = Beneficiary:Beneficiary.Postal
            */
            assertion = ((report.BeneficiaryPostalExists == true) && (report.BeneficiaryPostalEndDate != null && report.BeneficiaryPostalEndDate != report.BeneficiaryEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000229"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000229
    
            #region VR.ATO.CHTWTHHLDBEN.000229
    
            /*  VR.ATO.CHTWTHHLDBEN.000229
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.endDate WHERE CONTEXT(ALL) <> Beneficiary:period.endDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^EndDate1 <> NULL AND ^EndDate1 <> ^EndDate
    
            Data Elements:
    
            ^EndDate = Beneficiary
    
            ^EndDate1 = Beneficiary:Beneficiary.Residential
            */
            assertion = ((report.BeneficiaryResidentialExists == true) && (report.BeneficiaryResidentialEndDate != null && report.BeneficiaryResidentialEndDate != report.BeneficiaryEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.BeneficiaryOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLDBEN.000229"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLDBEN.000229
    
            #region VR.ATO.GEN.000459
    
            /*  VR.ATO.GEN.000459
            The Beneficiary Title code is not one of the valid codes
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Title.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.Title.Text <> SET(DOMAIN(TITLE CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN22 <> BLANK) AND (NotInDomain(^CHTWTHHLDBEN22, 'TITLE CODES'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN22 = Beneficiary:PersonNameDetails.Title.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN22) != true && !(IsMatch(report.CHTWTHHLDBEN22, @"^(2LT|AB|ABBOT|AC|ACM|ADM|AIR CDRE|ALDERMAN|AM|ARCHBISHOP|ARCHDEACON|ASSOC PROF|AVM|BARON|BARONESS|BISHOP|BR|BRIG|CANON|CAPT|CARDINAL|CDRE|CDT|CHAP|CMDR|CMM|COL|CONST|COUNT|COUNTESS|CPL|CPO|DAME|DEACON|DEACONESS|DEAN|DEPUTY SUPT|DR|DUCHESS|DUKE|EARL|EF|FLGOFF|FLT LT|FR|FSGT|GEN|GNR|GP CAPT|HON JUDGE|HON JUST|HON|HRH|INSP|JUDGE|JUST|LAC|LACW|LADY|LBDR|LCPL|LORD|LS|LT CMDR|LT COL|LT GEN|LT|MADAM|MAJ GEN|MAJ|MASTER|MATRON|MAYOR|MAYORESS|MIDN|MISS|MON|MOST REV|MR|MRS|MS|PASTOR|PATRIARCH|PLT OFF|PO|PRIOR|PROF|PTE|RABBI|RADM|RECTOR|REV|RF|RT HON|RT REV MON|RT REV|RTREV BISHOP|SBLT|SEN|SGT|SIR|SMN|SNR CONST|SQN LDR|SR|SSGT|SUPR|SWAMI|TF|VADM|VERY REV|VICAR|VISCOUNT|WG CDR|WO|WO1|WO2)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000459", Severity = ProcessMessageSeverity.Error,
                    Description = @"Title is invalid",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Title.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000459"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN22", Value = report.CHTWTHHLDBEN22 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000459
    
            #region VR.ATO.GEN.000470
    
            /*  VR.ATO.GEN.000470
            Title, first name, other given names or suffix is present without the family name.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Title.Text <> NULLORBLANK)
            AND (pyde.xx.xx:PersonNameDetails.FamilyName.Text = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN22 <> BLANK) AND (^CHTWTHHLDBEN23 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLDBEN22 = Beneficiary:PersonNameDetails.Title.Text
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN22) != true && string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN23) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000470", Severity = ProcessMessageSeverity.Error,
                    Description = @"Title is provided without Family Name",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Title.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000470"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN22", Value = report.CHTWTHHLDBEN22 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000470
    
            #region VR.ATO.GEN.000471
    
            /*  VR.ATO.GEN.000471
            Title, first name, other given names or suffix is present without the family name.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text  = NULLORBLANK)
            AND (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN23  = BLANK) AND (^CHTWTHHLDBEN24 <> BLANK)
    
            Data Elements:
    
            ^CHTWTHHLDBEN24 = Beneficiary:PersonNameDetails.GivenName.Text
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN23) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN24) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000471", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name is provided without Family name",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000471"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN24", Value = report.CHTWTHHLDBEN24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000471
    
            #region VR.ATO.GEN.000472
    
            /*  VR.ATO.GEN.000472
            Other given name/s present without the first name.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = NULLORBLANK)
            AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN24 = BLANK) AND (^CHTWTHHLDBEN25 <> BLANK)
    
            Data Elements:
    
            ^CHTWTHHLDBEN24 = Beneficiary:PersonNameDetails.GivenName.Text
    
            ^CHTWTHHLDBEN25 = Beneficiary:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN24) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN25) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000472", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name is required",
                    LongDescription = @"Other given name is provided without First name",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000472"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN24", Value = report.CHTWTHHLDBEN24 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN25", Value = report.CHTWTHHLDBEN25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000472
    
            #region VR.ATO.GEN.000475
    
            /*  VR.ATO.GEN.000475
            Title, first name, other given names or suffix is present without the family name.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = NULLORBLANK)
            AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text  <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN23 = BLANK) AND (^CHTWTHHLDBEN25  <> BLANK)
    
            Data Elements:
    
            ^CHTWTHHLDBEN25 = Beneficiary:PersonNameDetails.OtherGivenName.Text
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN23) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN25) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000475", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name is provided without Family name",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000475"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN25", Value = report.CHTWTHHLDBEN25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000475
    
            #region VR.ATO.GEN.410040
    
            /*  VR.ATO.GEN.410040
            Field contains all numeric characters with any combination of spaces; at least one alpha must be present.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.FamilyName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN23 <> BLANK) AND (NotContainsSet(^CHTWTHHLDBEN23, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN23) != true && !(IsMatch(report.CHTWTHHLDBEN23, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410040", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410040"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410040
    
            #region VR.ATO.GEN.410063
    
            /*  VR.ATO.GEN.410063
            Field contains all numeric characters with any combination of spaces; at least one alpha must be present.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN24 <> BLANK) AND (NotContainsSet(^CHTWTHHLDBEN24, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN24 = Beneficiary:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN24) != true && !(IsMatch(report.CHTWTHHLDBEN24, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063", Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410063"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN24", Value = report.CHTWTHHLDBEN24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410063
    
            #region VR.ATO.GEN.410131
    
            /*  VR.ATO.GEN.410131
            Field contains all numeric characters with any combination of spaces; at least one alpha must be present.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN25 <> BLANK) AND (NotContainsSet(^CHTWTHHLDBEN25, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN25 = Beneficiary:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN25) != true && !(IsMatch(report.CHTWTHHLDBEN25, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410131"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN25", Value = report.CHTWTHHLDBEN25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410131
    
            #region VR.ATO.GEN.410200
    
            /*  VR.ATO.GEN.410200
            Individual name field contains 'Exec for', 'Rep for' or 'Trustee for' at the beginning of the name field with a space after, in the field with a space before and after or at the end of the field with a space before. These abbreviations cannot be used.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN23, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN23, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410200"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410200
    
            #region VR.ATO.GEN.410201
    
            /*  VR.ATO.GEN.410201
            Individual name field contains titles MR, MRS, MISS or MS at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the field. These abbreviations cannot be used i
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN23, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN23, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410201"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410201
    
            #region VR.ATO.GEN.410202
    
            /*  VR.ATO.GEN.410202
            Individual name field contains suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the fi
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN24, '"ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN24 = Beneficiary:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN24, @"(^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | II | III | IV | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |II |III |IV |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| II| III| IV| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000436", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410202"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN24", Value = report.CHTWTHHLDBEN24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410202
    
            #region VR.ATO.GEN.410203
    
            /*  VR.ATO.GEN.410203
            Individual name field contains 'Exec for', 'Rep for' or 'Trustee for' at the beginning of the name field with a space after, in the field with a space before and after or at the end of the field with a space before. These abbreviations cannot be used.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN24, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN24 = Beneficiary:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN24, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410203"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN24", Value = report.CHTWTHHLDBEN24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410203
    
            #region VR.ATO.GEN.410204
    
            /*  VR.ATO.GEN.410204
            Individual name field contains titles MR, MRS, MISS or MS at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the field. These abbreviations cannot be used i
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN24, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN24 = Beneficiary:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN24, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410204"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN24", Value = report.CHTWTHHLDBEN24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410204
    
            #region VR.ATO.GEN.410207
    
            /*  VR.ATO.GEN.410207
            Individual name field contains suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the fi
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN25, '"ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN25 = Beneficiary:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN25, @"(^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | II | III | IV | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |II |III |IV |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| II| III| IV| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000448", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410207"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN25", Value = report.CHTWTHHLDBEN25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410207
    
            #region VR.ATO.GEN.410208
    
            /*  VR.ATO.GEN.410208
            Individual name field contains 'Exec for', 'Rep for' or 'Trustee for' at the beginning of the name field with a space after, in the field with a space before and after or at the end of the field with a space before. These abbreviations cannot be used.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN25, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN25 = Beneficiary:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN25, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410208"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN25", Value = report.CHTWTHHLDBEN25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410208
    
            #region VR.ATO.GEN.410209
    
            /*  VR.ATO.GEN.410209
            Individual name field contains titles MR, MRS, MISS or MS at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the field. These abbreviations cannot be used i
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN25, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN25 = Beneficiary:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN25, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410209"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN25", Value = report.CHTWTHHLDBEN25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410209
    
            #region VR.ATO.GEN.428036
    
            /*  VR.ATO.GEN.428036
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN70 <> BLANK) AND (NotInSet(^CHTWTHHLDBEN70, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN70 = Beneficiary.Business:AddressDetails.StateOrTerritory.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN70) != true && !(IsMatch(report.CHTWTHHLDBEN70, @"^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428036", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid State code",
                    LongDescription = @"The State code in the Current address is not Valid (Valid States are ""QLD "",""NSW "",""VIC "",""SA "",""WA "",""NT "",""ACT "",""TAS)",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428036"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN70", Value = report.CHTWTHHLDBEN70 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428036
    
            #region VR.ATO.GEN.428036
    
            /*  VR.ATO.GEN.428036
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN65 <> BLANK) AND (NotInSet(^CHTWTHHLDBEN65, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN65 = Beneficiary.Postal:AddressDetails.StateOrTerritory.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN65) != true && !(IsMatch(report.CHTWTHHLDBEN65, @"^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428036", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid State code",
                    LongDescription = @"The State code in the Current address is not Valid (Valid States are ""QLD "",""NSW "",""VIC "",""SA "",""WA "",""NT "",""ACT "",""TAS)",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428036"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN65", Value = report.CHTWTHHLDBEN65 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428036
    
            #region VR.ATO.GEN.428036
    
            /*  VR.ATO.GEN.428036
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN42 <> BLANK) AND (NotInSet(^CHTWTHHLDBEN42, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN42 = Beneficiary.Residential:AddressDetails.StateOrTerritory.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN42) != true && !(IsMatch(report.CHTWTHHLDBEN42, @"^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428036", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid State code",
                    LongDescription = @"The State code in the Current address is not Valid (Valid States are ""QLD "",""NSW "",""VIC "",""SA "",""WA "",""NT "",""ACT "",""TAS)",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428036"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN42", Value = report.CHTWTHHLDBEN42 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428036
    
            #region VR.ATO.GEN.428041
    
            /*  VR.ATO.GEN.428041
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Postcode.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Postcode.Text  <> SET(0200-9999))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN71 <> BLANK) AND (IsNumeric(^CHTWTHHLDBEN71) = 'FALSE' OR AsNumeric(^CHTWTHHLDBEN71) < 200 OR AsNumeric(^CHTWTHHLDBEN71) > 9999)
    
            Data Elements:
    
            ^CHTWTHHLDBEN71 = Beneficiary.Business:AddressDetails.Postcode.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN71) != true && (IsNumeric(report.CHTWTHHLDBEN71) == false || AsNumeric(report.CHTWTHHLDBEN71) < 200 || AsNumeric(report.CHTWTHHLDBEN71) > 9999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428041", Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, postcode is invalid. Valid range is 0200 - 9999",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Postcode.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428041"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN71", Value = report.CHTWTHHLDBEN71 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428041
    
            #region VR.ATO.GEN.428041
    
            /*  VR.ATO.GEN.428041
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Postcode.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Postcode.Text  <> SET(0200-9999))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN66 <> BLANK) AND (IsNumeric(^CHTWTHHLDBEN66) = 'FALSE' OR AsNumeric(^CHTWTHHLDBEN66) < 200 OR AsNumeric(^CHTWTHHLDBEN66) > 9999)
    
            Data Elements:
    
            ^CHTWTHHLDBEN66 = Beneficiary.Postal:AddressDetails.Postcode.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN66) != true && (IsNumeric(report.CHTWTHHLDBEN66) == false || AsNumeric(report.CHTWTHHLDBEN66) < 200 || AsNumeric(report.CHTWTHHLDBEN66) > 9999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428041", Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, postcode is invalid. Valid range is 0200 - 9999",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Postcode.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428041"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN66", Value = report.CHTWTHHLDBEN66 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428041
    
            #region VR.ATO.GEN.428041
    
            /*  VR.ATO.GEN.428041
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Postcode.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Postcode.Text  <> SET(0200-9999))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN43 <> BLANK) AND (IsNumeric(^CHTWTHHLDBEN43) = 'FALSE' OR AsNumeric(^CHTWTHHLDBEN43) < 200 OR AsNumeric(^CHTWTHHLDBEN43) > 9999)
    
            Data Elements:
    
            ^CHTWTHHLDBEN43 = Beneficiary.Residential:AddressDetails.Postcode.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN43) != true && (IsNumeric(report.CHTWTHHLDBEN43) == false || AsNumeric(report.CHTWTHHLDBEN43) < 200 || AsNumeric(report.CHTWTHHLDBEN43) > 9999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428041", Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, postcode is invalid. Valid range is 0200 - 9999",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Postcode.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428041"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN43", Value = report.CHTWTHHLDBEN43 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428041
    
            #region VR.ATO.GEN.428231
    
            /*  VR.ATO.GEN.428231
            Individual name field contains suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the fi
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLDBEN23, '"ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN23, @"(^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | II | III | IV | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |II |III |IV |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| II| III| IV| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000422", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428231"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428231

            #region VR.ATO.GEN.000408

            /*  VR.ATO.GEN.000408
            Address line 2 field cannot contain 'care of' reference.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN59 <> BLANK) AND (StartsWithSet(^CHTWTHHLDBEN59,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^CHTWTHHLDBEN59, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN59 = Beneficiary.Business:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN59) != true && (IsMatch(report.CHTWTHHLDBEN59, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.CHTWTHHLDBEN59, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN59", Value = report.CHTWTHHLDBEN59 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000408

            #region VR.ATO.GEN.000408

            /*  VR.ATO.GEN.000408
            Address line 2 field cannot contain 'care of' reference.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN63 <> BLANK) AND (StartsWithSet(^CHTWTHHLDBEN63,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^CHTWTHHLDBEN63, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN63 = Beneficiary.Postal:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN63) != true && (IsMatch(report.CHTWTHHLDBEN63, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.CHTWTHHLDBEN63, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN63", Value = report.CHTWTHHLDBEN63 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000408

            #region VR.ATO.GEN.000408

            /*  VR.ATO.GEN.000408
            Address line 2 field cannot contain 'care of' reference.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN40 <> BLANK) AND (StartsWithSet(^CHTWTHHLDBEN40,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^CHTWTHHLDBEN40, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN40 = Beneficiary.Residential:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN40) != true && (IsMatch(report.CHTWTHHLDBEN40, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.CHTWTHHLDBEN40, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN40", Value = report.CHTWTHHLDBEN40 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000408
    
            #region VR.ATO.GEN.428254
    
            /*  VR.ATO.GEN.428254
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN60, 'UNKNOWN')
    
            Data Elements:
    
            ^CHTWTHHLDBEN60 = Beneficiary.Business:AddressDetails.Line1.Text
            */
            assertion = (report.CHTWTHHLDBEN60 == null ? false : report.CHTWTHHLDBEN60.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN60", Value = report.CHTWTHHLDBEN60 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428254
    
            #region VR.ATO.GEN.428254
    
            /*  VR.ATO.GEN.428254
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN62, 'UNKNOWN')
    
            Data Elements:
    
            ^CHTWTHHLDBEN62 = Beneficiary.Postal:AddressDetails.Line1.Text
            */
            assertion = (report.CHTWTHHLDBEN62 == null ? false : report.CHTWTHHLDBEN62.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN62", Value = report.CHTWTHHLDBEN62 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428254
    
            #region VR.ATO.GEN.428254
    
            /*  VR.ATO.GEN.428254
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN39, 'UNKNOWN')
    
            Data Elements:
    
            ^CHTWTHHLDBEN39 = Beneficiary.Residential:AddressDetails.Line1.Text
            */
            assertion = (report.CHTWTHHLDBEN39 == null ? false : report.CHTWTHHLDBEN39.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN39", Value = report.CHTWTHHLDBEN39 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428254
    
            #region VR.ATO.GEN.428255
    
            /*  VR.ATO.GEN.428255
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN59, 'UNKNOWN')
    
            Data Elements:
    
            ^CHTWTHHLDBEN59 = Beneficiary.Business:AddressDetails.Line2.Text
            */
            assertion = (report.CHTWTHHLDBEN59 == null ? false : report.CHTWTHHLDBEN59.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN59", Value = report.CHTWTHHLDBEN59 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428255
    
            #region VR.ATO.GEN.428255
    
            /*  VR.ATO.GEN.428255
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN63, 'UNKNOWN')
    
            Data Elements:
    
            ^CHTWTHHLDBEN63 = Beneficiary.Postal:AddressDetails.Line2.Text
            */
            assertion = (report.CHTWTHHLDBEN63 == null ? false : report.CHTWTHHLDBEN63.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN63", Value = report.CHTWTHHLDBEN63 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428255
    
            #region VR.ATO.GEN.428255
    
            /*  VR.ATO.GEN.428255
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN40, 'UNKNOWN')
    
            Data Elements:
    
            ^CHTWTHHLDBEN40 = Beneficiary.Residential:AddressDetails.Line2.Text
            */
            assertion = (report.CHTWTHHLDBEN40 == null ? false : report.CHTWTHHLDBEN40.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN40", Value = report.CHTWTHHLDBEN40 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428255
    
            #region VR.ATO.GEN.428256
    
            /*  VR.ATO.GEN.428256
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN69, 'UNKNOWN')
    
            Data Elements:
    
            ^CHTWTHHLDBEN69 = Beneficiary.Business:AddressDetails.LocalityName.Text
            */
            assertion = (report.CHTWTHHLDBEN69 == null ? false : report.CHTWTHHLDBEN69.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryBusinessId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN69", Value = report.CHTWTHHLDBEN69 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428256
    
            #region VR.ATO.GEN.428256
    
            /*  VR.ATO.GEN.428256
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN64, 'UNKNOWN')
    
            Data Elements:
    
            ^CHTWTHHLDBEN64 = Beneficiary.Postal:AddressDetails.LocalityName.Text
            */
            assertion = (report.CHTWTHHLDBEN64 == null ? false : report.CHTWTHHLDBEN64.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryPostalId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN64", Value = report.CHTWTHHLDBEN64 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428256
    
            #region VR.ATO.GEN.428256
    
            /*  VR.ATO.GEN.428256
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN41, 'UNKNOWN')
    
            Data Elements:
    
            ^CHTWTHHLDBEN41 = Beneficiary.Residential:AddressDetails.LocalityName.Text
            */
            assertion = (report.CHTWTHHLDBEN41 == null ? false : report.CHTWTHHLDBEN41.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.BeneficiaryResidentialId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN41", Value = report.CHTWTHHLDBEN41 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428256
    
            #region VR.ATO.GEN.428258
    
            /*  VR.ATO.GEN.428258
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN32, ' - ')
    
            Data Elements:
    
            ^CHTWTHHLDBEN32 = Beneficiary:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.CHTWTHHLDBEN32 == null ? false : report.CHTWTHHLDBEN32.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN32", Value = report.CHTWTHHLDBEN32 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428258
    
            #region VR.ATO.GEN.428259
    
            /*  VR.ATO.GEN.428259
            Non-individual name field contains 'P/L' which is not a valid abbreviation. This abbreviation cannot be used.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS "P/L")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN32, 'P/L')
    
            Data Elements:
    
            ^CHTWTHHLDBEN32 = Beneficiary:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.CHTWTHHLDBEN32 == null ? false : report.CHTWTHHLDBEN32.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428259"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN32", Value = report.CHTWTHHLDBEN32 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428259
    
            #region VR.ATO.GEN.428260
    
            /*  VR.ATO.GEN.428260
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","'’","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^CHTWTHHLDBEN32, '"--","''","  "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN32 = Beneficiary:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN32, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN32", Value = report.CHTWTHHLDBEN32 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428260
    
            #region VR.ATO.GEN.428262
    
            /*  VR.ATO.GEN.428262
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN23, ' - ')
    
            Data Elements:
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.CHTWTHHLDBEN23 == null ? false : report.CHTWTHHLDBEN23.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428262"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428262
    
            #region VR.ATO.GEN.428263
    
            /*  VR.ATO.GEN.428263
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS SET("--","'’","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^CHTWTHHLDBEN23, '"--","''","  "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN23 = Beneficiary:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN23, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428263"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN23", Value = report.CHTWTHHLDBEN23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428263
    
            #region VR.ATO.GEN.428264
    
            /*  VR.ATO.GEN.428264
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN24, ' - ')
    
            Data Elements:
    
            ^CHTWTHHLDBEN24 = Beneficiary:PersonNameDetails.GivenName.Text
            */
            assertion = (report.CHTWTHHLDBEN24 == null ? false : report.CHTWTHHLDBEN24.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428264"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN24", Value = report.CHTWTHHLDBEN24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428264
    
            #region VR.ATO.GEN.428265
    
            /*  VR.ATO.GEN.428265
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS SET("--","'’","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^CHTWTHHLDBEN24, '"--","''","  "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN24 = Beneficiary:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN24, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428265"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN24", Value = report.CHTWTHHLDBEN24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428265
    
            #region VR.ATO.GEN.428266
    
            /*  VR.ATO.GEN.428266
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLDBEN25, ' - ')
    
            Data Elements:
    
            ^CHTWTHHLDBEN25 = Beneficiary:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.CHTWTHHLDBEN25 == null ? false : report.CHTWTHHLDBEN25.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428266"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN25", Value = report.CHTWTHHLDBEN25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428266
    
            #region VR.ATO.GEN.428267
    
            /*  VR.ATO.GEN.428267
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS SET("--","'’","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^CHTWTHHLDBEN25, '"--","''","  "'))
    
            Data Elements:
    
            ^CHTWTHHLDBEN25 = Beneficiary:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLDBEN25, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428267"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN25", Value = report.CHTWTHHLDBEN25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428267
    
            #region VR.ATO.GEN.434223
    
            /*  VR.ATO.GEN.434223
            The Australian Business Number (ABN) has failed the ABN algorithm check.
    
            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifer <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifer) = FALSE)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLDBEN33 <> BLANK) and (FailsABNAlgorithm(^CHTWTHHLDBEN33))
    
            Data Elements:
    
            ^CHTWTHHLDBEN33 = Beneficiary:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLDBEN33) != true && FailsABNAlgorithm(report.CHTWTHHLDBEN33));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434223"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLDBEN33", Value = report.CHTWTHHLDBEN33 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.434223

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }


    }
} 
